/**
 * \file        uart_cresnet_slave.hpp
 * \brief       low level uart class definition for cresnet and DMNet slave ports
 * \author      Larry Salant
 * \date        1/20/2009
 *
 */
#ifndef __DM_UART_CRESNET_SLAVE_HPP__
#define __DM_UART_CRESNET_SLAVE_HPP__

////////////////////////////////////////////////////////////////////////////////

#include "dm_nutypedefs.h"
#include "dm_cbuffer.h"
#include "Cresnet.h"
#include "dm_cresnet.h"
#include "CresnetStruct.h"
#include "DmEncapChecksumHelper.h"
#include "dm_NetMapper.hpp"

////////////////////////////////////////////////////////////////////////////////
// TSID is unique to card / product
void SetPNPNumber(UINT32 tsid);
UINT32 UartCresnetSlaveGetPPNAddr(void);
////////////////////////////////////////////////////////////////////////////////
// Hardware access routines
void UartCresnetSlaveGrabBus(UINT8 uart);
void UartCresnetSlaveDropBus(UINT8 uart);
extern "C" BOOL UartCresnetSlaveFifoIsEmpty(UINT8 uart);
extern "C" BOOL UartCresnetSlaveUartIsBusy(UINT8 uart);
void UartCresnetSlaveNetLed(BOOL on);
void ManageDMNET_LinkLed( void );

extern void (*pfProjectMidpointOnlineStatus)(UINT8);

////////////////////////////////////////////////////////////////////////////////

// make this configurable
#define CRESNET_RX_BUF_CNT              1
#define CRESENT_OUTQUEUE_LENGTH         4

typedef struct
{
    UINT8 autoid;
    UINT8 sendIdentity;
    CBuffer *pOutGoingCmdQueue;
}SLAVESTREAM;

typedef struct
{
    CBuffer *m_pPendingToController; // pointer to queue for controller
    UINT8 m_EndpointOnline;
}MIDPOINT_DATA;


typedef void (*UARTCRESNETFWDMSG)(UINT8 *packet);

#pragma pack()

class CDMUartCresnetSlave
{
public:
#ifndef REMOVE_DM_ENCAP_HELPER
    CDMUartCresnetSlave(UINT8 bNumStreams, UINT8 port, UINT8 uart, UINT8 timer, UINT8 cresnetId,UINT8 bRxEnableNeedCardReady = FALSE,
                        UINT8 outQueueLength = CRESENT_OUTQUEUE_LENGTH, UINT8 inMidpointQueueLength = CRESENT_OUTQUEUE_LENGTH, BOOL SendAutoID=TRUE,
                        DmEncapChecksumHelper* pHelper = 0, CDMNetMapper *tmpDMNetMapper = pDMNetMapper);
#else
    CDMUartCresnetSlave(UINT8 bNumStreams, UINT8 port, UINT8 uart, UINT8 timer, UINT8 cresnetId, UINT8 bRxEnableNeedCardReady = FALSE,
                        UINT8 outQueueLength = CRESENT_OUTQUEUE_LENGTH, UINT8 inMidpointQueueLength = CRESENT_OUTQUEUE_LENGTH, BOOL SendAutoID=TRUE,
                        void* pHelper = 0, CDMNetMapper *tmpDMNetMapper = pDMNetMapper);
#endif
    ~CDMUartCresnetSlave() ;
    UINT8 UartCresnetGetSlaveId( void );
    UINT8 IsUartCrestnetSlaveQueueFull(UINT8 bStream);
    UINT8 UartCresnetSlaveTxPacketReady(UINT8 bStream, UINT8 *pMessage);
    void UartCresnetSlaveTimerIsr(void);
    UINT8 UartCresnetSlaveGetID(void);
    void UartCresnetSlaveSetAutoId(UINT8 bStream);
    void UartCresnetSlaveSetIdentity(UINT8 bStream);
    INT32 UartCresnetSlaveQueueTxPacket(UINT8 bStream, void *packet, UINT32 packet_size);
    void UartCresnetSlaveGenerateBreak(void);
    UINT32 UartCresnetSlaveIsr(BOOL gotBreak, BOOL rxReady, UINT8 newByte, BOOL txEmpty);
    UINT32 UartCresnetSlaveRxIsr(void);
    void UartCresnetSlaveResetTxQueue(UINT8 bStream);

    // for plug and play
    void UartCresnetSlaveSetID(UINT8 ID);
    void UartCresnetSlaveSetRestartFlag(BOOL set);
    void UartCresnetSlaveSetTempID(UINT8 id);

    // for midpoint slave
    UINT32 UartCresnetMidpointRxIsr(void);
    CREST_ALL_PACKETS *MidpointGetNextPacket(void);
    void MidpointReleaseNextPacket(UINT8 *pBuffer);
    void MidpointOnlineTimer(void);
    void MidpointSetOnLineStatus(UINT8 State);
    void SendPacketToTxForEndpoint(UINT8 *pPacket);
    bool IsMidpointOnline(void);

    void StartRespondingToPolls()
    {
        m_bRespondToPoll = true;
    }

    void StopRespondingToPolls();   //Function definition moved to "uart_cresnet_slave.cpp"

    void SetBypassMapper( bool mode )
    {
        m_bypassMapper = mode;
    }
    void SetFwdMsgFunctionPointer( UARTCRESNETFWDMSG pfUartCresnetFwdMsg )
    {
        m_pfUartCresnetFwdMsg = pfUartCresnetFwdMsg;
    }

    void Reconfigure( BOOL bEnable );

    //Make public for debug purposes
    SLAVESTREAM *m_pStream;
    MIDPOINT_DATA *m_MidData;
    CDMNetMapper *m_pDMNetMapper;

    UINT8 m_uart;       // instance of hardware uart to use
    UINT8 m_timer;      // instance of hardware timer to use
    UINT8 m_port;       // indicates if this instance is for Controller or Midpoint port
    UINT32 m_bitsPerSec;
    UINT32 m_charTimeUsec;
    // +1 is to leave room for the source byte
    UINT8 m_rxData[CRESNET_RX_BUF_CNT][CNET_MAX_MSG_SIZE+1];
    UINT32 m_rxDataIdx;
    UINT32 m_rxWriteBuf;
    UINT8 m_netmode;
    UINT8 m_netmodeNext;
    UINT8 m_id;
    UINT8 m_bNumStreams;   // number of consective cresnet id's (1/stream)starting at id
    UINT16 m_txBytes;
    UINT16 m_txPos;
    UINT8 * m_sendPtr;
    BOOL m_restart;
    UINT8 m_immediatePkt[CNET_MAX_MSG_SIZE];
    BOOL m_pktInProgress;

    // data needed for Midpoint Slave
    UINT8 OfflineTimeout;

    BOOL        m_bRespondToPoll;               // Indicates whether we should respond to POLLS



private:
    void UartCresnetSlaveSendPacket(UINT8 id);
    void UartCresnetSlaveTimeNet(UINT32 usec);
    void UartCresnetSlaveBuildIdMsg(UINT8 bStream);
    void UartCresnetSlaveBuildUpdateRequest(void);
    int UartCresnetSlaveValidateLocalMsg(UINT8 *packet);
    UINT8 UartCresnetSlaveIsOurID(UINT8 id);
    INT32 UartCresnetSlaveGetNextTxPacket(UINT8 bStream);
    void UartCresnetSlaveWaitAfterBreak(void);
    void UartCresnetSlaveResetTimer(void);
    void UartCresnetBreak(UINT32 inst, BOOL begin);
    void UartCresnetSlaveSetNetmode(UINT8 netmode);
    INT32 UartCresnetSlaveSendToTxQueue(UINT8 bStream, UINT8 * packet);
    void UartCresnetSlaveTx(UINT8 data);
    void UartCresnetSlaveBuildIdentityResponse(UINT8 bStream);
    INT32 UartCresnetSlavePPNImmediateProcessing(UINT8 * packet, UINT8 id, BOOL * pRecursed, BOOL * pReturn);
    void UartCresnetSlaveDisableReceiver(void);
    void UartCresnetSlaveEnableReceiver(void);
    void UartCresnetSlaveFlushReceiver(void);

#ifndef REMOVE_DM_ENCAP_HELPER
    //DmEncapChecksumHelper class
    DmEncapChecksumHelper* m_pEncapChecksumHelper;
#endif

#ifdef CNET_STATS
    UINT32 m_isrCnt;
    UINT32 m_timerCnt;
    UINT32 m_breakCnt;
    UINT32 m_rxByteCnt;
    UINT32 m_txByteCnt;
    UINT32 m_txReadyCnt;
    UINT32 m_pollCnt;
    UINT32 m_overrunCnt;
#endif
    BOOL   m_sync;
    BOOL   m_bypassMapper;
    UARTCRESNETFWDMSG m_pfUartCresnetFwdMsg;

    BOOL m_bTxEnable;
};

#define MAX_SLAVE_PORTS 2

extern CDMUartCresnetSlave *pUartCresnetSlave[MAX_SLAVE_PORTS];
CDMUartCresnetSlave *GetCresnetSlaveUart(void);
CDMUartCresnetSlave *GetDMMidpointUart(UINT8 streamId);

// offsets into Midpoint buffer (CNET packt with encapsulation)
#define MIDPKT_START_ENCAP_LEN 1
#define MIDPKT_START_UNENCAP   3  // start of packet received

#endif
